<?php



if ( ! defined( 'ABSPATH' ) ) {

	exit; // Exit if accessed directly.
}

/**
 * Main Marino Core Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */

final class Marino_Extension {

	/**
	 * Plugin Version
	 *
	 * @since 1.0.0
	 *
	 * @var string The plugin version.
	 */
	const VERSION = '1.0.0';

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */

	const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.0';


	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Elementor_Test_Extension The single instance of the class.
	 */

	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Elementor_Test_Extension An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}

	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {
		add_action( 'plugins_loaded', [ $this, 'init' ] );
	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

		// Check if Elementor installed and activated

		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return;
		}

		// Check for required Elementor version

		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return;
		}

		// Check for required PHP version

		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return;
		}


		// Add Plugin actions

		add_action( 'elementor/widgets/register', [ $this, 'init_widgets' ] );

        // Register widget scripts

		add_action( 'elementor/frontend/after_enqueue_scripts', [ $this, 'widget_scripts' ]);

		// Specific Register widget scripts
		add_action( 'elementor/frontend/after_register_scripts', [ $this, 'marino_regsiter_widget_scripts' ] );

        // category register

		add_action( 'elementor/elements/categories_registered',[ $this, 'marino_elementor_widget_categories' ] );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'marino' ),
			'<strong>' . esc_html__( 'Marino Core', 'marino' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'marino' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */

			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'marino' ),
			'<strong>' . esc_html__( 'Marino Core', 'marino' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'marino' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}
	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		if ( isset( $_GET['activate'] ) ) unset( $_GET['activate'] );

		$message = sprintf(

			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'marino' ),
			'<strong>' . esc_html__( 'Marino Core', 'marino' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'marino' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */

	public function init_widgets() {

		// Include Widget files

		require_once( MARINO_ADDONS . '/widgets/section-title.php' );
		require_once( MARINO_ADDONS . '/widgets/button.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-blog.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-brand.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-testimonials.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-gallery.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-hero-slider.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-hero-video.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-services.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-image.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-image-box.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-counter.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-hunting-spot.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-team-member.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-fishing-spot.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-product.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-working-step.php' );
		require_once( MARINO_ADDONS . '/widgets/marino-contact-info.php' );

		// Header Elements

		require_once( MARINO_ADDONS . '/header/new-search.php' );
		require_once( MARINO_ADDONS . '/header/header-info.php' );
		require_once( MARINO_ADDONS . '/header/header-menu.php' );
		require_once( MARINO_ADDONS . '/header/marino-mobile-menu.php' );
		require_once( MARINO_ADDONS . '/header/cart-count.php' );
		require_once( MARINO_ADDONS . '/header/offcanvas.php' );
		require_once( MARINO_ADDONS . '/header/marino-gtranslate.php' );

		// Register widget

		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Section_Title_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Testimonial_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Client_Logo_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Blog_Post() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Button_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Gallery_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Hero_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Hero_Video() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Service_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Image() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Image_Box() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Counterup() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Huntar_Spot_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Team_Member() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Fishing_Spot_Widget() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Product_Slider() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Woking_Step() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Contact_Info() );
		

		// Header Widget Register

		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Header_Search() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Header_Info() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Header_Menu() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Mobilemenu() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Cart_Count() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Offcanvas() );
		\Elementor\Plugin::instance()->widgets_manager->register( new \Marino_Gtranslate() );
	}

    public function widget_scripts() {

        wp_enqueue_script(
            'marino-frontend-script',
            MARINO_PLUGDIRURI . 'assets/js/marino-frontend.js',
            array('jquery'),
            false,
            true
		);
	}
	
	public function marino_regsiter_widget_scripts( ) {
		
		wp_register_script(
            'counter-up',
            MARINO_PLUGDIRURI . 'assets/js/jquery.counterup.min.js',
            array('jquery'),
            '1.0',
            true
		);


	}


    function marino_elementor_widget_categories( $elements_manager ) {

        $elements_manager->add_category(
            'marino',
            [
                'title' => __( 'Marino', 'marino' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );

        $elements_manager->add_category(
            'marino_footer_elements',
            [
                'title' => __( 'Marino Footer Elements', 'marino' ),
                'icon' 	=> 'fa fa-plug',
            ]
		);

		$elements_manager->add_category(
            'marino_header_elements',
            [
                'title' => __( 'Marino Header Elements', 'marino' ),
                'icon' 	=> 'fa fa-plug',
            ]
        );
	}
}

Marino_Extension::instance();